/************************************************************************************
* Implements PD and PLME PHY primitives
*
* (c) Copyright 2009, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
************************************************************************************/


/************************************************************************************
*************************************************************************************
* Includes
*************************************************************************************
************************************************************************************/
#include "EmbeddedTypes.h"
#include "PortConfig.h"
#include "Interrupt.h"
#include "Phy.h"
#include "TransceiverDrv.h"
#include "PhyPibExtended.h"
#include "SMAC_Config.h"


uint8_t *gpPhyTxPacket;

 
/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

/******************************************************************************
*Function name: PhyDataRequest
*
*Parameters passed in: phyPacket_t *pTxPacket, uint8_t phyTxMode
*
*Return type: uint8_t
*
*Prototype: uint8_t PhyDataRequest(phyPacket_t *pTxPacket, uint8_t phyTxMode);
*
*Description: Implements the setup for a TX request and initiates data transmission. 
*             Returns gPhyBusy_c if the software PHY is found in a different state 
*             than gIdle_c and no operation is performed, otherwise returns gPhySuccess_c.
*             This operation will be started immediately if phyTxMode is gImmediateAction_c
*             otherwise is delayed until trigger timer expires, depending on a previous 
*             call to PhyTimeSetEventTrigger()
******************************************************************************/

uint8_t PhyDataRequest(phyPacket_t *pTxPacket, uint8_t phyTxMode)
{
  uint8_t fifoThreshold = gFifoThreshold_c, CRC;	
    
  gpPhyTxPacket = pTxPacket->data;
  // null pointer
  if(NULL == pTxPacket)
  {
    return gPhyInvalidParam_c;
  }
  // cannot have packets shorter than gPhyMinDataLength_c
  if((pTxPacket->frameLength < gPhyMinDataLength_c) || (pTxPacket->frameLength > gPhyMaxDataLength_c))
  {
    return gPhyInvalidParam_c;
  }
 
  if( gIdle_c != PhyGetState() )
  {
    return gPhyBusy_c;
  }
    /* disable transceiver interrupts */
   //Disable DIO0,DIO1 and DIO4 port interrupts
  NVIC_ICER = 0x80000000;
 
  /* pass TX PHY parameters to the transceiver ISR */
  fifoThreshold = PhyPassTxParams(pTxPacket);
  
  /* set FIFO threshold and Tx start condition */
  MKW01Drv_WriteRegister(MKW01_Reg_FifoThresh, (fifoThreshold & 0x7F));
  
  /* write PHR in FIFO */
  MKW01Drv_WriteRegister(MKW01_Reg_Fifo, pTxPacket->frameLength );
  
#if defined (IAR)  
  CRC = IntDisableAll();	
#elif  defined (CW)
  IntDisableAll(CRC);		
#endif 
  
  /* Pre-fill FIFO */
  MKW01Drv_FillFifo();
  IntRestoreAll(CRC) ;
  
  /* put PHY in TX state */
  PhySetState(gTX_c);
  
  gpfPendingTxEvent = PhyTxHandleFifoLevelEvent;
  
  /* PacketSent -> DIO0 (PTDD PIN 7), FifoLevel -> DIO1 (PTDD PIN 6) */
    MKW01Drv_ConfigureDIOPins((DIO0_TxPacketSent | DIO1_TxFifoLevel | DIO2_TxLowBat | DIO3_TxLowBat), (DIO4_TxLowBat | DIO5_TxClkOut));  
    
  /* enable PacketSent (PTDD PIN 7) interrupt */ 
  /* enable FifoLevel (PTDD PIN 6) interrupt */
    MKW01Drv_IRQ_Enable(9, 0xA);                                                //DIO0, Rising Edge and DIO1, Falling edge Enabled
    //Enable PortC and PortD Interrupts
    NVIC_ICPR |= 0x80000000;
    NVIC_ISER |= 0x80000000;
 
  if(phyTxMode == gImmediateAction_c)
  {
    //PhyTimeDisableEventTrigger();
    /* Put transceiver in TX mode */ 
    MKW01Drv_WriteRegister(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_Transmitter)); //TX should be enabled until this line, and the Interrupt Flag set.
  }
  
  return gPhySuccess_c;
}


/******************************************************************************
*Function name: PhyPlmeRxRequest
*
*Parameters passed in: phyPacket_t *pRxData, phyRxParams_t *pRxParams, uint8_t phyRxMode
*
*Return type: uint8_t
*
*Prototype: uint8_t PhyPlmeRxRequest(phyPacket_t *pRxData,phyRxParams_t *pRxParams,uint8_t phyRxMode);
*
*Description:  Implements the setup for a RX request action. Returns gPhyBusy_c if 
*              the software PHY is found in a different state than gIdle_c and no 
*              operation is performed, otherwise returns gPhySuccess_c. This operation 
*              will be started immediately if phyTxMode is gImmediateAction_c otherwise 
*              is delayed until trigger timer expires, depending on a previous 
*              call to PhyTimeSetEventTrigger().
******************************************************************************/

uint8_t  PhyPlmeRxRequest(phyPacket_t *pRxData, phyRxParams_t *pRxParams, uint8_t phyRxMode)
{
   if( (NULL == pRxData) || (NULL == pRxParams))
  {
    return gPhyInvalidParam_c;
  }

  if( gIdle_c != PhyGetState() )
  {
    return gPhyBusy_c;
  }
  
 /* disable transceiver interrupts */
   //Disable DIO0,DIO1 and DIO4 port interrupts
  NVIC_ICER = 0x80000000;
  
  //set FIFO threshold and Tx start condition
  MKW01Drv_WriteRegister(MKW01_Reg_FifoThresh, (gFifoThreshold_c & 0x7F));
  
  //configure RSSI threshols
  //MKW01Drv_WriteRegister(MKW01_Reg_RssiThresh, gSmacRssiThreshold_c);         
  
  //pass RX PHY parameters to the transceiver ISR
  //PhyPassRxParams(pRxData, pRxParams, (uint8_t) (phyRxMode & gRxAckAction));
  PhyPassRxParams(pRxData, pRxParams, 0);               // ACK disabled
 
  //put PHY in RX state
  PhySetState(gRX_c);
  
  //SyncAddres -> DIO0 (KBI2 PIN 0), FifoLevel -> DIO1 (KBI2 PIN 1), Timeout -> DIO4 (KBI2 PIN 4)
  MKW01Drv_ConfigureDIOPins((DIO0_RxSyncAddress | DIO1_RxFifoLevel | DIO2_RxLowBat | DIO3_RxFifoFull),(DIO4_RxTimeout | DIO5_RxClkOut));
  
  /* set FIFO threshold an Tx start condition */
  MKW01Drv_WriteRegister(MKW01_Reg_FifoThresh, 0);
  
  gpfPendingRxEvent = PhyRxHandleSyncAddresEvent;
   
  //configure SyncAddress (PTDD PIN 7) as interrupt (rising edge) (DIO0)
  //configure FifoLevel (PTDD PIN 6) as interrupt (rising edge)   (DIO1)
   MKW01Drv_IRQ_Enable(9,9);                                                    //TODO, create macros instead of hardnumbers for sensitivity
  
    //Re-enable PORTC and PORTD interrupts
   NVIC_ICPR |= 0x80000000;
   NVIC_ISER |= 0x80000000;
  
  if((phyRxMode & gImmediateAction_c))
  {
    //PhyTimeDisableEventTrigger();
    //Put transceiver in RX mode
    MKW01Drv_WriteRegister(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_Receiver )); //RX
  }

  return gPhySuccess_c;
}

/******************************************************************************
*Function name: PhyPlmeCcaEdRequest
*
*Parameters passed in: uint8_t ccaType, uint8_t phyCcaMode
*
*Return type: uint8_t
*
*Prototype: uint8_t  PhyPlmeCcaEdRequest(uint8_t ccaType, uint8_t phyCcaMode);
*
*Description: Implements the setup for a CCA request if ccaType is gCcaCCA_MODE1_c
*             or ED request if ccaType is gCcaED_c. Returns gPhyBusy_c if the software
*             PHY is found in a different state than gIdle_c and no operation is 
*             performed, otherwise returns gPhySuccess_c. This operation will be started
*             immediately if phyCcaMode is gImmediateAction_c otherwise is delayed until 
*             trigger timer expires, depending on a previous call to PhyTimeSetEventTrigger().
******************************************************************************/

uint8_t PhyPlmeCcaEdRequest(uint8_t ccaParam, uint8_t phyCcaMode)
{
 
  // only CCA mode1 or gCcaED_c 
  if((gCcaCCA_MODE1_c != ccaParam) && (gCcaED_c != ccaParam) )
  {
    return gPhyInvalidParam_c;
  }

  if( gIdle_c != PhyGetState() )
  {
    return gPhyBusy_c;
  }
  
  /* disable transceiver interrupts */
  //Disable DIO0,DIO1 and DIO4 port interrupts
  NVIC_ICER = 0x80000000;

  //configure RSSI threshols
  MKW01Drv_WriteRegister(MKW01_Reg_RssiThresh, (uint8_t) (0xFF)); // -127.5 dBm
  
  //configure LNA gain
  MKW01Drv_WriteRegister(MKW01_Reg_Lna, (uint8_t) ( Lna_LnaZin_50 | Lna_LnaGain_Agc ) );
  
  //pass CCA PHY parameters to the transceiver ISR
  PhyPassCcaParams(ccaParam);
  
  //put PHY in CCA or ED state
  if(gCcaCCA_MODE1_c == ccaParam)
  {
    PhySetState(gCCA_c);
  }
  else
  {
    PhySetState(gED_c);
  }
  
  //Rssi -> DIO0 (PTD7), FifoLevel -> DIO1 (PTD6), Timeout -> DIO4 (KBI2 PIN 4)
  MKW01Drv_ConfigureDIOPins((DIO0_RxRssi | DIO1_RxFifoNotEmpty | DIO2_RxLowBat | DIO3_RxFifoFull),(DIO4_RxRxReady | DIO5_RxClkOut));

   //configure RxReady (PTD4) as interrupt (rising edge)
  MKW01Drv_CCA_Enable();  
  //Re-enable the PORTC and PORTD interrupts
  NVIC_ICPR |= 0x80000000;
  NVIC_ISER |= 0x80000000;
  
  if(phyCcaMode == gImmediateAction_c)
  {
   // PhyTimeDisableEventTrigger();
    //Put transceiver in RX mode
    MKW01Drv_WriteRegister(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_Receiver )); //RX
  }
  return gPhySuccess_c;
}

/******************************************************************************
*Function name: PhyPlmeForceTrxOffRequest
*
*Parameters passed in: none
*
*Return type: none
*
*Prototype: void    PhyPlmeForceTrxOffRequest(void);
*
*Description: Abort the current requested action; put the PHY in the idle state, 
              put the transceiver in stand-by mode. Disables any previous 
              timeout eventually programmed by calling PhyTimeDisableEventTimeout().
******************************************************************************/

void PhyPlmeForceTrxOffRequest(void)
{

  
 // PhyTimeDisableEventTrigger();
 // PhyTimeDisableEventTimeout();
  
  if( gIdle_c != PhyGetState() )
  {
    /* disable transceiver interrupts */                                        
    NVIC_ICER = 0x80000000;                                                     //Disable DIO0,DIO1 and DIO4 port interrupts
    MKW01Drv_IRQ_Disable();                                                     //Disable DIO0 and DIO1 IRQ Interrupts

    gpfPendingRxEvent = PhyRxHandleDummyEvent;
    
    gpfPendingTxEvent = PhyTxHandleDummyEvent;
    
    PhySetState(gIdle_c);
    
    //Put transceiver in Stand-By mode
    MKW01Drv_WriteRegister(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_StandBy )); //StandBy;
    
    //clear the transceiver FIFO
    //while FIFO not empty
    while(MKW01Drv_ReadRegister(MKW01_Reg_IrqFlags2) & 0x40)
    {
      (void)MKW01Drv_ReadRegister(MKW01_Reg_Fifo);
    }
    
 
    //Re-enable the PORTC and PORTD interrupts                                  //PortC and portD (same vector for both ports on L2KM)
    NVIC_ICPR |= 0x80000000;
    NVIC_ISER |= 0x80000000;
  }

}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyPlmeWaitRequest(void) {
}

/******************************************************************************
*Function name: InitializePhy
*Parameters passed in: none
*Return type: none
*Prototype:
*void    InitializePhy(void)
*Description: This function performs the PHY layer initialization: 
*             initialize PHY layer's software parameters, initialize the 
*             transceiver with default registers values, starts PHY timer, 
*             initialize PHY PIB structure with default values.
******************************************************************************/

void InitializePhy(void) {
  
  MKW01Drv_RFdefaultInit();  
  PhyPib_InitOnStartUp();  
 // PhyTimerStart();
  //PhyEventTimeoutInit();
 // PhyEventTriggerInit();
  MKW01xDrv_IRQ_PortConfig();                                                   
  MKW01Drv_CCA_Enable();                                                        
 
}
